<?php

namespace App\Http\Controllers\Tenant;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Spatie\Activitylog\Models\Activity;
use App\Models\User;
use Carbon\Carbon;

class ActivityLogController extends Controller
{
    /**
     * Display a paginated list of activity logs for the current tenant.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        // Authorize the request
        $this->authorize('view-activity-log');
        
        // Build the query for tenant-scoped activity logs
        $query = Activity::query()
            ->where('tenant_id', auth()->user()->tenant_id)
            ->with('causer')
            ->latest();
        
        // Apply filters if provided
        if ($request->filled('causer_id')) {
            $query->where('causer_id', $request->causer_id);
        }
        
        if ($request->filled('subject_type')) {
            $query->where('subject_type', 'LIKE', '%' . $request->subject_type . '%');
        }
        
        if ($request->filled('event')) {
            $query->where('event', $request->event);
        }
        
        if ($request->filled('log_name')) {
            $query->where('log_name', $request->log_name);
        }
        
        if ($request->filled('date_from')) {
            $dateFrom = Carbon::parse($request->date_from)->startOfDay();
            $query->where('created_at', '>=', $dateFrom);
        }
        
        if ($request->filled('date_to')) {
            $dateTo = Carbon::parse($request->date_to)->endOfDay();
            $query->where('created_at', '<=', $dateTo);
        }
        
        // Paginate results
        $logs = $query->paginate(15)->withQueryString();
        
        // Get users for filter dropdown
        $users = User::where('tenant_id', auth()->user()->tenant_id)
            ->select('id', 'name')
            ->orderBy('name')
            ->get();
        
        // Get distinct subject types for filter dropdown
        $subjectTypes = Activity::where('tenant_id', auth()->user()->tenant_id)
            ->select('subject_type')
            ->distinct()
            ->whereNotNull('subject_type')
            ->get()
            ->pluck('subject_type')
            ->map(function ($type) {
                // Extract only the model name from namespace
                $parts = explode('\\', $type);
                return end($parts);
            });
        
        // Get distinct events for filter dropdown
        $events = Activity::where('tenant_id', auth()->user()->tenant_id)
            ->select('event')
            ->distinct()
            ->whereNotNull('event')
            ->orderBy('event')
            ->pluck('event');
            
        // Return Inertia view with data
        return Inertia::render('Tenant/ActivityLog/Index', [
            'logs' => $logs,
            'filters' => $request->only(['causer_id', 'subject_type', 'event', 'log_name', 'date_from', 'date_to']),
            'users' => $users,
            'subjectTypes' => $subjectTypes,
            'events' => $events,
        ]);
    }
}
