<?php

namespace App\Http\Controllers\Superadmin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Spatie\Activitylog\Models\Activity;
use App\Models\User;
use App\Models\Tenant;
use Carbon\Carbon;

class ActivityLogController extends Controller
{
    /**
     * Display a paginated list of activity logs for superadmins.
     * Allows viewing logs for all tenants or filtering by tenant.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Inertia\Response
     */
    public function index(Request $request)
    {
        // Authorize the request - only superadmins can access
        $this->authorize('access-superadmin-panel');
        
        // Build the query for all activity logs
        $query = Activity::query()
            ->with('causer')
            ->latest();
        
        // Apply filters if provided
        if ($request->filled('tenant_id')) {
            $query->where('tenant_id', $request->tenant_id);
        }
        
        if ($request->filled('causer_id')) {
            $query->where('causer_id', $request->causer_id);
        }
        
        if ($request->filled('subject_type')) {
            $query->where('subject_type', 'LIKE', '%' . $request->subject_type . '%');
        }
        
        if ($request->filled('event')) {
            $query->where('event', $request->event);
        }
        
        if ($request->filled('log_name')) {
            $query->where('log_name', $request->log_name);
        }
        
        if ($request->filled('date_from')) {
            $dateFrom = Carbon::parse($request->date_from)->startOfDay();
            $query->where('created_at', '>=', $dateFrom);
        }
        
        if ($request->filled('date_to')) {
            $dateTo = Carbon::parse($request->date_to)->endOfDay();
            $query->where('created_at', '<=', $dateTo);
        }
        
        // Include platform level logs (null tenant_id)
        if ($request->filled('platform_only') && $request->platform_only === 'true') {
            $query->whereNull('tenant_id');
        }
        
        // Paginate results
        $logs = $query->paginate(15)->withQueryString();
        
        // Get all tenants for filter dropdown
        $tenants = Tenant::select('id', 'name')->orderBy('name')->get();
        
        // Get users for filter dropdown - limit to 100 most active users for performance
        $users = User::select('id', 'name', 'tenant_id')
            ->with('tenant:id,name')
            ->orderBy('name')
            ->limit(100)
            ->get()
            ->map(function ($user) {
                // Add tenant name to display name for better identification
                return [
                    'id' => $user->id,
                    'name' => $user->name . ($user->tenant ? ' (' . $user->tenant->name . ')' : ' (Platform)'),
                ];
            });
        
        // Get distinct subject types for filter dropdown
        $subjectTypes = Activity::select('subject_type')
            ->distinct()
            ->whereNotNull('subject_type')
            ->get()
            ->pluck('subject_type')
            ->map(function ($type) {
                // Extract only the model name from namespace
                $parts = explode('\\', $type);
                return end($parts);
            });
        
        // Get distinct events for filter dropdown
        $events = Activity::select('event')
            ->distinct()
            ->whereNotNull('event')
            ->orderBy('event')
            ->pluck('event');
            
        // Return Inertia view with data
        return Inertia::render('Superadmin/ActivityLog/Index', [
            'logs' => $logs,
            'filters' => $request->only([
                'tenant_id', 'causer_id', 'subject_type', 'event', 
                'log_name', 'date_from', 'date_to', 'platform_only'
            ]),
            'tenants' => $tenants,
            'users' => $users,
            'subjectTypes' => $subjectTypes,
            'events' => $events,
        ]);
    }
}
