<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreSaleReturnRequest;
use App\Models\Client;
use App\Models\Product;
use App\Models\Sale;
use App\Models\SaleItem;
use App\Models\SaleReturn;
use App\Models\SaleReturnItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class SaleReturnController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('permission:process sales returns');
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = SaleReturn::where('tenant_id', Auth::user()->tenant_id);
        
        // Filter by client if provided
        if ($request->has('client_id')) {
            $query->where('client_id', $request->client_id);
        }
        
        // Filter by sale if provided
        if ($request->has('sale_id')) {
            $query->where('sale_id', $request->sale_id);
        }
        
        $returns = $query->with(['client:id,name', 'sale:id', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return response()->json($returns);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StoreSaleReturnRequest $request, string $saleId)
    {
        return DB::transaction(function () use ($request, $saleId) {
            // Find the sale and lock for update (exclude soft deleted)
            $sale = Sale::withoutTrashed()
                ->where('id', $saleId)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->with('saleItems')
                ->lockForUpdate()
                ->first();
                
            if (!$sale) {
                return response()->json(['message' => 'Sale not found'], 404);
            }
            
            // Find the client and lock for update
            $client = Client::where('id', $sale->client_id)
                ->lockForUpdate()
                ->first();
                
            if (!$client) {
                return response()->json(['message' => 'Client not found'], 404);
            }
            
            // Create the sale return record
            $saleReturn = new SaleReturn([
                'tenant_id' => Auth::user()->tenant_id,
                'sale_id' => $sale->id,
                'client_id' => $client->id,
                'user_id' => Auth::id(),
                'reason' => $request->reason,
                'returned_amount' => 0, // Will be calculated from items
                'status' => 'completed',
            ]);
            
            $saleReturn->save();
            
            $returnedAmount = 0;
            
            // Process each return item
            foreach ($request->items as $item) {
                // Find the sale item and lock for update
                $saleItem = SaleItem::where('id', $item['sale_item_id'])
                    ->where('sale_id', $sale->id)
                    ->lockForUpdate()
                    ->first();
                    
                if (!$saleItem) {
                    throw new \Exception("Sale item not found: {$item['sale_item_id']}");
                }
                
                // Find the product and lock for update
                $product = Product::where('id', $saleItem->product_id)
                    ->lockForUpdate()
                    ->first();
                    
                if (!$product) {
                    throw new \Exception("Product not found: {$saleItem->product_id}");
                }
                
                $quantity = $item['quantity'];
                $priceAtSale = $saleItem->price_at_sale; // Use the original price
                $subtotal = $quantity * $priceAtSale;
                
                // Create the return item
                $returnItem = new SaleReturnItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'sale_return_id' => $saleReturn->id,
                    'product_id' => $product->id,
                    'sale_item_id' => $saleItem->id,
                    'quantity' => $quantity,
                    'price_at_sale' => $priceAtSale,
                    'subtotal' => $subtotal,
                ]);
                
                $returnItem->save();
                
                // Update product quantity (increase stock)
                $product->quantity += $quantity;
                $product->save();
                
                $returnedAmount += $subtotal;
            }
            
            // Update the return total
            $saleReturn->returned_amount = $returnedAmount;
            $saleReturn->save();
            
            // Update client balance (client owes less)
            $client->balance -= $returnedAmount;
            $client->save();
            
            // Update sale status if all items are returned
            $totalReturnedItems = $sale->returns()
                ->with('items')
                ->get()
                ->flatMap(function ($return) {
                    return $return->items;
                })
                ->groupBy('sale_item_id')
                ->map(function ($items, $saleItemId) {
                    return $items->sum('quantity');
                });
                
            $allItemsReturned = true;
            foreach ($sale->saleItems as $saleItem) {
                $returnedQty = $totalReturnedItems[$saleItem->id] ?? 0;
                if ($returnedQty < $saleItem->quantity) {
                    $allItemsReturned = false;
                    break;
                }
            }
            
            if ($allItemsReturned) {
                $sale->status = 'returned';
                $sale->save();
            }
            
            // Load the items relationship for the response
            $saleReturn->load('items.product');
            
            return response()->json($saleReturn, 201);
        });
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $saleReturn = SaleReturn::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with(['client', 'sale', 'items.product', 'user'])
            ->first();
            
        if (!$saleReturn) {
            return response()->json(['message' => 'Sale return not found'], 404);
        }
        
        return response()->json($saleReturn);
    }
}
