<?php

namespace App\Http\Controllers;

use App\Models\Client;
use App\Models\Expense;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchaseReturn;
use App\Models\Sale;
use App\Models\SaleReturn;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class ReportController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        // Use permission middleware with the correct format
        $this->middleware('permission:view reports');
    }
    
    /**
     * Generate a sales report.
     */
    public function salesReport(Request $request)
    {
        // Set default date range to current month before validation
        if (!$request->has('start_date') || !$request->start_date) {
            $request->merge(['start_date' => now()->startOfMonth()->format('Y-m-d')]);
        }
        
        if (!$request->has('end_date') || !$request->end_date) {
            $request->merge(['end_date' => now()->endOfMonth()->format('Y-m-d')]);
        }
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'client_id' => 'nullable|exists:clients,id,tenant_id,' . Auth::user()->tenant_id,
            'group_by' => 'nullable|in:product,date,client',
        ]);
        
        // Use withoutTrashed() to exclude soft-deleted sales
        $query = Sale::withoutTrashed()->where('tenant_id', Auth::user()->tenant_id);
        
        // Apply filters
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        if ($request->filled('client_id')) {
            $query->where('client_id', $request->client_id);
        }
        
        // Group by if specified
        if ($request->group_by === 'product') {
            $report = DB::table('sale_items')
                ->join('sales', 'sale_items.sale_id', '=', 'sales.id')
                ->join('products', 'sale_items.product_id', '=', 'products.id')
                ->where('sales.tenant_id', Auth::user()->tenant_id)
                ->whereNull('sales.deleted_at')
                ->select(
                    'products.id',
                    'products.name',
                    DB::raw('SUM(sale_items.quantity) as total_quantity'),
                    DB::raw('SUM(sale_items.quantity * sale_items.price_at_sale) as total_amount')
                )
                ->groupBy('products.id', 'products.name')
                ->orderBy('total_amount', 'desc')
                ->get();
        } elseif ($request->group_by === 'date') {
            $report = DB::table('sales')
                ->where('tenant_id', Auth::user()->tenant_id)
                ->whereNull('deleted_at')
                ->select(
                    DB::raw('DATE(created_at) as date'),
                    DB::raw('COUNT(*) as total_sales'),
                    DB::raw('SUM(final_amount) as total_amount')
                )
                ->groupBy('date')
                ->orderBy('date', 'desc')
                ->get();
        } elseif ($request->group_by === 'client') {
            $report = DB::table('sales')
                ->join('clients', 'sales.client_id', '=', 'clients.id')
                ->where('sales.tenant_id', Auth::user()->tenant_id)
                ->whereNull('sales.deleted_at')
                ->select(
                    'clients.id',
                    'clients.name',
                    DB::raw('COUNT(*) as total_sales'),
                    DB::raw('SUM(sales.final_amount) as total_amount')
                )
                ->groupBy('clients.id', 'clients.name')
                ->orderBy('total_amount', 'desc')
                ->get();
        } else {
            // Default: list all sales with basic info
            $sales = $query->with(['client:id,name', 'user:id,name'])
                ->orderBy('created_at', 'desc')
                ->paginate(15);
                
            return response()->json($sales);
        }
        
        return response()->json($report);
    }
    
    /**
     * Generate an inventory report.
     */
    public function inventoryReport(Request $request)
    {
        $request->validate([
            'low_stock_only' => 'nullable|boolean',
        ]);
        
        $query = Product::where('tenant_id', Auth::user()->tenant_id);
        
        // Filter for low stock items if requested
        if ($request->filled('low_stock_only') && $request->low_stock_only) {
            $query->whereRaw('quantity <= reorder_point');
        }
        
        $products = $query->select([
                'id', 'name', 'sku', 'barcode', 'price', 'purchase_price',
                'quantity', 'reorder_point',
                DB::raw('(quantity <= reorder_point) as low_stock')
            ])
            ->orderBy('low_stock', 'desc')
            ->orderBy('quantity', 'asc')
            ->get();
            
        // Calculate inventory value
        $totalInventoryValue = $products->sum(function ($product) {
            return $product->quantity * $product->purchase_price;
        });
        
        return response()->json([
            'products' => $products,
            'total_inventory_value' => $totalInventoryValue,
            'low_stock_count' => $products->where('low_stock', true)->count(),
        ]);
    }
    
    /**
     * Generate a client statement.
     */
    public function clientStatement(string $clientId)
    {
        $client = Client::where('id', $clientId)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$client) {
            return response()->json(['message' => 'Client not found'], 404);
        }
        
        // Get sales for this client (exclude soft deleted)
        $sales = Sale::withoutTrashed()
            ->where('client_id', $client->id)
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Get sale returns for this client
        $saleReturns = SaleReturn::where('client_id', $client->id)
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Get payments for this client
        $payments = $client->payments()
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Combine all transactions and sort by date
        $transactions = collect();
        
        foreach ($sales as $sale) {
            $transactions->push([
                'id' => $sale->id,
                'type' => 'sale',
                'date' => $sale->created_at,
                'amount' => $sale->final_amount,
                'balance_effect' => $sale->final_amount, // Increases client balance (they owe more)
                'description' => 'Sale #' . $sale->id,
                'user' => $sale->user ? $sale->user->name : null,
            ]);
        }
        
        foreach ($saleReturns as $return) {
            $transactions->push([
                'id' => $return->id,
                'type' => 'sale_return',
                'date' => $return->created_at,
                'amount' => $return->returned_amount,
                'balance_effect' => -$return->returned_amount, // Decreases client balance (they owe less)
                'description' => 'Return for Sale #' . $return->sale_id,
                'user' => $return->user ? $return->user->name : null,
            ]);
        }
        
        foreach ($payments as $payment) {
            $description = 'Payment (' . ucfirst($payment->method) . ')';
            
            // Add status information for check payments
            if ($payment->isCheckPayment()) {
                $description .= ' - Check #' . $payment->check_number;
                if ($payment->check_status === 'unpaid') {
                    $description .= ' (Unpaid)';
                } elseif ($payment->check_status === 'pending') {
                    $description .= ' (Pending)';
                } elseif ($payment->check_status === 'paid') {
                    $description .= ' (Paid)';
                }
            }
            
            $transactions->push([
                'id' => $payment->id,
                'type' => 'payment',
                'date' => $payment->created_at,
                'amount' => $payment->amount,
                'balance_effect' => $payment->shouldCountInTotals() ? -$payment->amount : 0,
                'description' => $description,
                'user' => $payment->user ? $payment->user->name : null,
                'status' => $payment->isCheckPayment() ? $payment->check_status : 'completed',
                'is_valid' => $payment->shouldCountInTotals(),
            ]);
        }
        
        $transactions = $transactions->sortByDesc('date')->values();
        
        return response()->json([
            'client' => $client,
            'transactions' => $transactions,
            'current_balance' => $client->balance,
            'total_sales' => $sales->sum('final_amount'),
            'total_returns' => $saleReturns->sum('returned_amount'),
            'total_payments' => $payments->filter(function ($payment) {
                return $payment->shouldCountInTotals();
            })->sum('amount'),
        ]);
    }
    
    /**
     * Generate a supplier statement.
     */
    public function supplierStatement(string $supplierId)
    {
        $supplier = Supplier::where('id', $supplierId)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$supplier) {
            return response()->json(['message' => 'Supplier not found'], 404);
        }
        
        // Get purchases for this supplier
        $purchases = Purchase::where('supplier_id', $supplier->id)
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Get purchase returns for this supplier
        $purchaseReturns = PurchaseReturn::where('supplier_id', $supplier->id)
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Get payments for this supplier
        $payments = $supplier->payments()
            ->with('user:id,name')
            ->orderBy('created_at', 'desc')
            ->get();
            
        // Combine all transactions and sort by date
        $transactions = collect();
        
        foreach ($purchases as $purchase) {
            $transactions->push([
                'id' => $purchase->id,
                'type' => 'purchase',
                'date' => $purchase->created_at,
                'amount' => $purchase->total_amount,
                'balance_effect' => $purchase->total_amount, // Increases supplier balance (we owe more)
                'description' => 'Purchase #' . $purchase->id,
                'user' => $purchase->user ? $purchase->user->name : null,
            ]);
        }
        
        foreach ($purchaseReturns as $return) {
            $transactions->push([
                'id' => $return->id,
                'type' => 'purchase_return',
                'date' => $return->created_at,
                'amount' => $return->returned_amount,
                'balance_effect' => -$return->returned_amount, // Decreases supplier balance (we owe less)
                'description' => 'Return for Purchase #' . $return->purchase_id,
                'user' => $return->user ? $return->user->name : null,
            ]);
        }
        
        foreach ($payments as $payment) {
            $description = 'Payment (' . ucfirst($payment->method) . ')';
            
            // Add status information for check payments
            if ($payment->isCheckPayment()) {
                $description .= ' - Check #' . $payment->check_number;
                if ($payment->check_status === 'unpaid') {
                    $description .= ' (Unpaid)';
                } elseif ($payment->check_status === 'pending') {
                    $description .= ' (Pending)';
                } elseif ($payment->check_status === 'paid') {
                    $description .= ' (Paid)';
                }
            }
            
            $transactions->push([
                'id' => $payment->id,
                'type' => 'payment',
                'date' => $payment->created_at,
                'amount' => $payment->amount,
                'balance_effect' => $payment->shouldCountInTotals() ? -$payment->amount : 0,
                'description' => $description,
                'user' => $payment->user ? $payment->user->name : null,
                'status' => $payment->isCheckPayment() ? $payment->check_status : 'completed',
                'is_valid' => $payment->shouldCountInTotals(),
            ]);
        }
        
        $transactions = $transactions->sortByDesc('date')->values();
        
        return response()->json([
            'supplier' => $supplier,
            'transactions' => $transactions,
            'current_balance' => $supplier->balance,
            'total_purchases' => $purchases->sum('total_amount'),
            'total_returns' => $purchaseReturns->sum('returned_amount'),
            'total_payments' => $payments->filter(function ($payment) {
                return $payment->shouldCountInTotals();
            })->sum('amount'),
        ]);
    }
    
    /**
     * Generate an expense report.
     */
    public function expenseReport(Request $request)
    {
        // Set default date range to current month before validation
        if (!$request->has('start_date') || !$request->start_date) {
            $request->merge(['start_date' => now()->startOfMonth()->format('Y-m-d')]);
        }
        
        if (!$request->has('end_date') || !$request->end_date) {
            $request->merge(['end_date' => now()->endOfMonth()->format('Y-m-d')]);
        }
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'category' => 'nullable|string',
            'group_by' => 'nullable|in:category,date',
        ]);
        
        $query = Expense::where('tenant_id', Auth::user()->tenant_id);
        
        // Apply filters
        if ($request->filled('start_date')) {
            $query->whereDate('expense_date', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->whereDate('expense_date', '<=', $request->end_date);
        }
        
        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }
        
        // Group by if specified
        if ($request->group_by === 'category') {
            $report = $query->select(
                    'category',
                    DB::raw('COUNT(*) as total_count'),
                    DB::raw('SUM(amount) as total_amount')
                )
                ->groupBy('category')
                ->orderBy('total_amount', 'desc')
                ->get();
        } elseif ($request->group_by === 'date') {
            $report = $query->select(
                    DB::raw('DATE(expense_date) as date'),
                    DB::raw('COUNT(*) as total_count'),
                    DB::raw('SUM(amount) as total_amount')
                )
                ->groupBy('date')
                ->orderBy('date', 'desc')
                ->get();
        } else {
            // Default: list all expenses with basic info
            $expenses = $query->with('user:id,name')
                ->orderBy('expense_date', 'desc')
                ->paginate(15);
                
            return response()->json($expenses);
        }
        
        // Get total amount for the filtered expenses
        $totalAmount = $query->sum('amount');
        
        return response()->json([
            'report' => $report,
            'total_amount' => $totalAmount,
        ]);
    }
    
    /**
     * Generate a purchase report.
     */
    public function purchaseReport(Request $request)
    {
        // Set default date range to current month before validation
        if (!$request->has('start_date') || !$request->start_date) {
            $request->merge(['start_date' => now()->startOfMonth()->format('Y-m-d')]);
        }
        
        if (!$request->has('end_date') || !$request->end_date) {
            $request->merge(['end_date' => now()->endOfMonth()->format('Y-m-d')]);
        }
        
        $request->validate([
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'supplier_id' => 'nullable|exists:suppliers,id,tenant_id,' . Auth::user()->tenant_id,
            'group_by' => 'nullable|in:product,date,supplier',
        ]);
        
        // Use withoutTrashed() to exclude soft-deleted purchases
        $query = Purchase::withoutTrashed()->where('tenant_id', Auth::user()->tenant_id);
        
        // Apply filters
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }
        
        // Group by if specified
        if ($request->group_by === 'product') {
            $report = DB::table('purchase_items')
                ->join('purchases', 'purchase_items.purchase_id', '=', 'purchases.id')
                ->join('products', 'purchase_items.product_id', '=', 'products.id')
                ->where('purchases.tenant_id', Auth::user()->tenant_id)
                ->whereNull('purchases.deleted_at')
                ->select(
                    'products.id',
                    'products.name',
                    DB::raw('SUM(purchase_items.quantity) as total_quantity'),
                    DB::raw('SUM(purchase_items.quantity * purchase_items.cost_price) as total_amount')
                )
                ->groupBy('products.id', 'products.name')
                ->orderBy('total_amount', 'desc')
                ->get();
        } elseif ($request->group_by === 'date') {
            $report = DB::table('purchases')
                ->where('tenant_id', Auth::user()->tenant_id)
                ->whereNull('deleted_at')
                ->select(
                    DB::raw('DATE(created_at) as date'),
                    DB::raw('COUNT(*) as total_purchases'),
                    DB::raw('SUM(total_amount) as total_amount')
                )
                ->groupBy('date')
                ->orderBy('date', 'desc')
                ->get();
        } elseif ($request->group_by === 'supplier') {
            $report = DB::table('purchases')
                ->join('suppliers', 'purchases.supplier_id', '=', 'suppliers.id')
                ->where('purchases.tenant_id', Auth::user()->tenant_id)
                ->whereNull('purchases.deleted_at')
                ->select(
                    'suppliers.id',
                    'suppliers.name',
                    DB::raw('COUNT(*) as total_purchases'),
                    DB::raw('SUM(purchases.total_amount) as total_amount')
                )
                ->groupBy('suppliers.id', 'suppliers.name')
                ->orderBy('total_amount', 'desc')
                ->get();
        } else {
            // Default: list all purchases with basic info
            $purchases = $query->with(['supplier:id,name', 'user:id,name'])
                ->orderBy('created_at', 'desc')
                ->paginate(15);
                
            return response()->json($purchases);
        }
        
        return response()->json($report);
    }
    
    /**
     * Generate a sales returns report.
     */
    public function salesReturnsReport(Request $request)
    {
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'client_id' => 'nullable|exists:clients,id,tenant_id,' . Auth::user()->tenant_id,
        ]);
        
        $query = SaleReturn::where('tenant_id', Auth::user()->tenant_id);
        
        // Apply filters
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        if ($request->filled('client_id')) {
            $query->where('client_id', $request->client_id);
        }
        
        $returns = $query->with(['client:id,name', 'sale:id', 'user:id,name', 'items.product:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        // Get total amount for the filtered returns
        $totalAmount = $query->sum('returned_amount');
        
        return response()->json([
            'returns' => $returns,
            'total_amount' => $totalAmount,
        ]);
    }
    
    /**
     * Generate a purchase returns report.
     */
    public function purchaseReturnsReport(Request $request)
    {
        $request->validate([
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
            'supplier_id' => 'nullable|exists:suppliers,id,tenant_id,' . Auth::user()->tenant_id,
        ]);
        
        $query = PurchaseReturn::where('tenant_id', Auth::user()->tenant_id);
        
        // Apply filters
        if ($request->filled('start_date')) {
            $query->whereDate('created_at', '>=', $request->start_date);
        }
        
        if ($request->filled('end_date')) {
            $query->whereDate('created_at', '<=', $request->end_date);
        }
        
        if ($request->filled('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }
        
        $returns = $query->with(['supplier:id,name', 'purchase:id', 'user:id,name', 'items.product:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        // Get total amount for the filtered returns
        $totalAmount = $query->sum('returned_amount');
        
        return response()->json([
            'returns' => $returns,
            'total_amount' => $totalAmount,
        ]);
    }
}
