<?php

namespace App\Http\Controllers;

use App\Http\Requests\StorePurchaseReturnRequest;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\PurchaseReturn;
use App\Models\PurchaseReturnItem;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PurchaseReturnController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('permission:process purchase returns');
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $query = PurchaseReturn::where('tenant_id', Auth::user()->tenant_id);
        
        // Filter by supplier if provided
        if ($request->has('supplier_id')) {
            $query->where('supplier_id', $request->supplier_id);
        }
        
        // Filter by purchase if provided
        if ($request->has('purchase_id')) {
            $query->where('purchase_id', $request->purchase_id);
        }
        
        $returns = $query->with(['supplier:id,name', 'purchase:id', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return response()->json($returns);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StorePurchaseReturnRequest $request, string $purchaseId)
    {
        return DB::transaction(function () use ($request, $purchaseId) {
            // Find the purchase and lock for update
            $purchase = Purchase::where('id', $purchaseId)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->with('items')
                ->lockForUpdate()
                ->first();
                
            if (!$purchase) {
                return response()->json(['message' => 'Purchase not found'], 404);
            }
            
            // Find the supplier and lock for update
            $supplier = Supplier::where('id', $purchase->supplier_id)
                ->lockForUpdate()
                ->first();
                
            if (!$supplier) {
                return response()->json(['message' => 'Supplier not found'], 404);
            }
            
            // Create the purchase return record
            $purchaseReturn = new PurchaseReturn([
                'tenant_id' => Auth::user()->tenant_id,
                'purchase_id' => $purchase->id,
                'supplier_id' => $supplier->id,
                'user_id' => Auth::id(),
                'reason' => $request->reason,
                'returned_amount' => 0, // Will be calculated from items
                'status' => 'completed',
            ]);
            
            $purchaseReturn->save();
            
            $returnedAmount = 0;
            
            // Process each return item
            foreach ($request->items as $item) {
                // Find the purchase item and lock for update
                $purchaseItem = PurchaseItem::where('id', $item['purchase_item_id'])
                    ->where('purchase_id', $purchase->id)
                    ->lockForUpdate()
                    ->first();
                    
                if (!$purchaseItem) {
                    throw new \Exception("Purchase item not found: {$item['purchase_item_id']}");
                }
                
                // Find the product and lock for update
                $product = Product::where('id', $purchaseItem->product_id)
                    ->lockForUpdate()
                    ->first();
                    
                if (!$product) {
                    throw new \Exception("Product not found: {$purchaseItem->product_id}");
                }
                
                $quantity = $item['quantity'];
                $costPrice = $purchaseItem->cost_price; // Use the original cost price
                $subtotal = $quantity * $costPrice;
                
                // Create the return item
                $returnItem = new PurchaseReturnItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'purchase_return_id' => $purchaseReturn->id,
                    'product_id' => $product->id,
                    'purchase_item_id' => $purchaseItem->id,
                    'quantity' => $quantity,
                    'cost_price' => $costPrice,
                    'subtotal' => $subtotal,
                ]);
                
                $returnItem->save();
                
                // Update product quantity (decrease stock)
                $product->quantity = max(0, $product->quantity - $quantity);
                $product->save();
                
                $returnedAmount += $subtotal;
            }
            
            // Update the return total
            $purchaseReturn->returned_amount = $returnedAmount;
            $purchaseReturn->save();
            
            // Update supplier balance (we owe the supplier less)
            $supplier->balance -= $returnedAmount;
            $supplier->save();
            
            // Load the items relationship for the response
            $purchaseReturn->load('items.product');
            
            return response()->json($purchaseReturn, 201);
        });
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $purchaseReturn = PurchaseReturn::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with(['supplier', 'purchase', 'items.product', 'user'])
            ->first();
            
        if (!$purchaseReturn) {
            return response()->json(['message' => 'Purchase return not found'], 404);
        }
        
        return response()->json($purchaseReturn);
    }
}
