<?php

namespace App\Http\Controllers;

use App\Http\Requests\StorePurchaseRequest;
use App\Http\Requests\UpdatePurchaseRequest;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PurchaseController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('permission:manage purchases')->except(['index', 'show']);
    }
    
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $purchases = Purchase::withoutTrashed() // Exclude soft-deleted purchases
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with(['supplier:id,name', 'user:id,name'])
            ->orderBy('created_at', 'desc')
            ->paginate(15);
            
        return response()->json($purchases);
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(StorePurchaseRequest $request)
    {
        return DB::transaction(function () use ($request) {
            // Find the supplier and lock for update to prevent race conditions
            $supplier = Supplier::where('id', $request->supplier_id)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->first();
                
            if (!$supplier) {
                return response()->json(['message' => 'Supplier not found'], 404);
            }
            
            // Create the purchase record
            $purchase = new Purchase([
                'tenant_id' => Auth::user()->tenant_id,
                'supplier_id' => $supplier->id,
                'user_id' => Auth::id(),
                'total_amount' => 0, // Will be calculated from items
                'status' => $request->status ?? 'completed',
                'notes' => $request->notes,
            ]);
            
            $purchase->save();
            
            $totalAmount = 0;
            
            // Process each purchase item
            foreach ($request->items as $item) {
                // Find the product and lock for update
                $product = Product::where('id', $item['product_id'])
                    ->where('tenant_id', Auth::user()->tenant_id)
                    ->lockForUpdate()
                    ->first();
                    
                if (!$product) {
                    throw new \Exception("Product not found: {$item['product_id']}");
                }
                
                $quantity = $item['quantity'];
                $costPrice = $item['cost_price'];
                $subtotal = $quantity * $costPrice;
                
                // Create the purchase item
                $purchaseItem = new PurchaseItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'purchase_id' => $purchase->id,
                    'product_id' => $product->id,
                    'quantity' => $quantity,
                    'cost_price' => $costPrice,
                    'subtotal' => $subtotal,
                ]);
                
                $purchaseItem->save();
                
                // Update product quantity and purchase price
                $product->quantity += $quantity;
                
                // Update the product's purchase price with the new cost price
                $product->purchase_price = $costPrice;
                
                $product->save();
                
                $totalAmount += $subtotal;
            }
            
            // Update the purchase total
            $purchase->total_amount = $totalAmount;
            $purchase->save();
            
            // Update supplier balance (we owe the supplier more)
            $supplier->balance += $totalAmount;
            $supplier->save();
            
            // Load the items relationship for the response
            $purchase->load('items.product');
            
            return response()->json($purchase, 201);
        });
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $purchase = Purchase::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with(['supplier', 'items.product', 'user'])
            ->first();
            
        if (!$purchase) {
            return response()->json(['message' => 'Purchase not found'], 404);
        }
        
        return response()->json($purchase);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdatePurchaseRequest $request, string $id)
    {
        // Only status and notes can be updated after creation
        $purchase = Purchase::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$purchase) {
            return response()->json(['message' => 'Purchase not found'], 404);
        }
        
        if ($request->filled('status')) {
            $purchase->status = $request->status;
        }
        
        if ($request->filled('notes')) {
            $purchase->notes = $request->notes;
        }
        
        $purchase->save();
        
        return response()->json($purchase);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $purchase = Purchase::where('id', $id)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->with('items')
            ->first();
            
        if (!$purchase) {
            return response()->json(['message' => 'Purchase not found'], 404);
        }
        
        // Check if the purchase has any returns
        $hasReturns = $purchase->returns()->exists();
        if ($hasReturns) {
            return response()->json([
                'message' => 'Cannot delete purchase with associated returns'
            ], 422);
        }
        
        return DB::transaction(function () use ($purchase) {
            // Find the supplier and lock for update
            $supplier = Supplier::where('id', $purchase->supplier_id)
                ->lockForUpdate()
                ->first();
                
            if (!$supplier) {
                return response()->json(['message' => 'Supplier not found'], 404);
            }
            
            // Revert product quantities
            foreach ($purchase->items as $item) {
                $product = Product::where('id', $item->product_id)
                    ->lockForUpdate()
                    ->first();
                    
                if ($product) {
                    // Ensure we don't go below zero
                    $newQuantity = max(0, $product->quantity - $item->quantity);
                    $product->quantity = $newQuantity;
                    $product->save();
                }
            }
            
            // Update supplier balance (we owe the supplier less)
            $supplier->balance -= $purchase->total_amount;
            $supplier->save();
            
            // Delete the purchase and its items (cascade)
            $purchase->delete();
            
            return response()->json(null, 204);
        });
    }
}
