<?php

namespace App\Http\Controllers;

use App\Http\Requests\StoreClientPaymentRequest;
use App\Http\Requests\StoreSupplierPaymentRequest;
use App\Models\Client;
use App\Models\Payment;
use App\Models\Supplier;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class PaymentController extends Controller
{
    /**
     * Store a new payment for a client.
     */
    public function storeClientPayment(StoreClientPaymentRequest $request, string $clientId)
    {
        return DB::transaction(function () use ($request, $clientId) {
            // Find the client and lock for update to prevent race conditions
            $client = Client::where('id', $clientId)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->first();
                
            if (!$client) {
                return response()->json(['message' => 'Client not found'], 404);
            }
            
            // Create the payment record
            $payment = new Payment([
                'tenant_id' => Auth::user()->tenant_id,
                'paymentable_type' => Client::class,
                'paymentable_id' => $client->id,
                'user_id' => Auth::id(),
                'amount' => $request->amount,
                'method' => $request->method,
                'payment_date' => $request->payment_date,
                'notes' => $request->notes,
            ]);
            
            // Add method-specific fields
            if ($request->method === 'bank_transfer') {
                $payment->bank_transfer_reference = $request->bank_transfer_reference;
                $payment->bank_transfer_receipt_path = $request->bank_transfer_receipt_path;
            } else if ($request->method === 'check') {
                $payment->check_number = $request->check_number;
                $payment->check_due_date = $request->check_due_date;
            }
            
            // Update client balance (client owes less)
            // For check payments, only reduce balance if the check should count in totals
            if ($payment->shouldCountInTotals()) {
                $client->balance -= $request->amount;
                $client->save();
            }
            
            // Save the payment
            $payment->save();
            
            return response()->json($payment, 201);
        });
    }
    
    /**
     * Store a new payment for a supplier.
     */
    public function storeSupplierPayment(StoreSupplierPaymentRequest $request, string $supplierId)
    {
        return DB::transaction(function () use ($request, $supplierId) {
            // Find the supplier and lock for update to prevent race conditions
            $supplier = Supplier::where('id', $supplierId)
                ->where('tenant_id', Auth::user()->tenant_id)
                ->lockForUpdate()
                ->first();
                
            if (!$supplier) {
                return response()->json(['message' => 'Supplier not found'], 404);
            }
            
            // Create the payment record
            $payment = new Payment([
                'tenant_id' => Auth::user()->tenant_id,
                'paymentable_type' => Supplier::class,
                'paymentable_id' => $supplier->id,
                'user_id' => Auth::id(),
                'amount' => $request->amount,
                'method' => $request->method,
                'payment_date' => $request->payment_date,
                'notes' => $request->notes,
            ]);
            
            // Add method-specific fields
            if ($request->method === 'bank_transfer') {
                $payment->bank_transfer_reference = $request->bank_transfer_reference;
                $payment->bank_transfer_receipt_path = $request->bank_transfer_receipt_path;
            } else if ($request->method === 'check') {
                $payment->check_number = $request->check_number;
                $payment->check_due_date = $request->check_due_date;
            }
            
            // Update supplier balance (supplier is owed less / we owe less)
            // For check payments, only reduce balance if the check should count in totals
            if ($payment->shouldCountInTotals()) {
                $supplier->balance -= $request->amount;
                $supplier->save();
            }
            
            // Save the payment
            $payment->save();
            
            return response()->json($payment, 201);
        });
    }
    
    /**
     * Get payments for a client.
     */
    public function getClientPayments(string $clientId)
    {
        // Find the client
        $client = Client::where('id', $clientId)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$client) {
            return response()->json(['message' => 'Client not found'], 404);
        }
        
        // Get the payments with pagination
        $payments = $client->payments()
            ->with('user')
            ->orderBy('payment_date', 'desc')
            ->paginate(15);
            
        return response()->json($payments);
    }
    
    /**
     * Get payments for a supplier.
     */
    public function getSupplierPayments(string $supplierId)
    {
        // Find the supplier
        $supplier = Supplier::where('id', $supplierId)
            ->where('tenant_id', Auth::user()->tenant_id)
            ->first();
            
        if (!$supplier) {
            return response()->json(['message' => 'Supplier not found'], 404);
        }
        
        // Get the payments with pagination
        $payments = $supplier->payments()
            ->with('user')
            ->orderBy('payment_date', 'desc')
            ->paginate(15);
            
        return response()->json($payments);
    }
}
