<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Inertia\Inertia;
use App\Models\Sale;
use App\Models\Client;
use App\Models\Supplier;
use App\Models\Product;
use App\Models\Expense;
use App\Models\Payment;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Create a new controller instance.
     */
    public function __construct()
    {
        $this->middleware('auth');
    }
    
    /**
     * Show the application dashboard.
     *
     * @return \Inertia\Response
     */
    public function index()
    {
        $user = Auth::user();
        
        // Check if user is superadmin and render a different view
        if ($user->is_superadmin) {
            // Data for superadmin minimal dashboard
            $tenantCount = \App\Models\Tenant::count();
            
            return Inertia::render('Superadmin/DashboardIndex', [
                'tenantCount' => $tenantCount,
                // Add any other minimal platform stats here later
            ]);
        }
        
        $tenantId = $user->tenant_id;
        $today = Carbon::today();
        
        // Get today's sales with their items (excluding soft-deleted sales)
        $todaySales = Sale::with('saleItems')
            ->withoutTrashed() // Exclude soft-deleted sales
            ->where('tenant_id', $tenantId)
            ->whereDate('created_at', $today)
            ->get();
            
        // Calculate total amount for each sale if needed and sum up the totals
        $totalSalesToday = 0;
        foreach ($todaySales as $sale) {
            if ($sale->final_amount == 0 && $sale->saleItems->count() > 0) {
                // Update the sale with the calculated total
                $calculatedTotal = $sale->calculateTotalAmount();
                $sale->total_amount = $calculatedTotal;
                // Set final_amount to total_amount if no discount was applied
                if ($sale->discount_amount == 0) {
                    $sale->final_amount = $calculatedTotal;
                }
                $sale->save();
                $totalSalesToday += $sale->final_amount;
            } else {
                $totalSalesToday += $sale->final_amount;
            }
        }
            
        // Get low stock products for all users
        $lowStockProducts = Product::where('tenant_id', $tenantId)
            ->whereNotNull('reorder_point')
            ->whereRaw('quantity <= reorder_point')
            ->orderBy('quantity', 'asc')
            ->limit(10)
            ->get(['id', 'name', 'quantity', 'reorder_point']);
            
        // Get due checks for all users
        $dueChecks = Payment::getDueChecks()->map(function ($check) {
            return [
                'id' => $check->id,
                'amount' => $check->amount,
                'check_number' => $check->check_number,
                'check_due_date' => $check->check_due_date,
                'paymentable_name' => $check->paymentable ? $check->paymentable->name : 'Unknown',
                'days_overdue' => now()->diffInDays($check->check_due_date, false),
            ];
        });
            
        // Base dashboard data for all users
        $dashboardData = [
            'totalSalesToday' => $totalSalesToday,
            'salesCountToday' => $todaySales->count(),
            'lowStockProducts' => $lowStockProducts,
            'lowStockItemsCount' => $lowStockProducts->count(),
            'dueChecks' => $dueChecks,
            'dueChecksCount' => $dueChecks->count(),
        ];
        
        // Additional data for admin/manager roles
        if ($user->can('view-reports')) {
            // Outstanding client balances
            $outstandingClientBalances = Client::where('tenant_id', $tenantId)
                ->where('balance', '>', 0)
                ->sum('balance');
            
            // Owed supplier balances
            $owedSupplierBalances = Supplier::where('tenant_id', $tenantId)
                ->where('balance', '>', 0)
                ->sum('balance');
            
            // Get out of stock products
            $outOfStockProducts = Product::where('tenant_id', $tenantId)
                ->where('quantity', '<=', 0)
                ->limit(5)
                ->get();
            
            // Calculate FIFO inventory valuation
            $inventoryValuation = 0;
            $inventoryLayers = \App\Models\InventoryLayer::whereHas('product', function($query) use ($tenantId) {
                $query->where('tenant_id', $tenantId);
            })->where('quantity_remaining', '>', 0)->get();
            
            foreach ($inventoryLayers as $layer) {
                $inventoryValuation += $layer->quantity_remaining * $layer->unit_cost;
            }
            
            // Get top 5 products by inventory value
            $topProductsByValue = Product::where('tenant_id', $tenantId)
                ->where('quantity', '>', 0)
                ->withCount(['inventoryLayers as inventory_value' => function($query) {
                    $query->select(\DB::raw('SUM(quantity_remaining * unit_cost)'));
                }])
                ->orderBy('inventory_value', 'desc')
                ->limit(5)
                ->get();
            
            
            // Recent expenses (last 7 days)
            $recentExpensesTotal = Expense::where('tenant_id', $tenantId)
                ->where('expense_date', '>=', Carbon::now()->subDays(7))
                ->sum('amount');
            
            // Add admin/manager specific data
            $dashboardData = array_merge($dashboardData, [
                'outstandingClientBalances' => $outstandingClientBalances,
                'owedSupplierBalances' => $owedSupplierBalances,
                'outOfStockProducts' => $outOfStockProducts,
                'inventoryValuation' => $inventoryValuation,
                'topProductsByValue' => $topProductsByValue,
                'recentExpensesTotal' => $recentExpensesTotal,
                'recentExpenses' => Expense::where('tenant_id', $tenantId)
                    ->where('expense_date', '>=', Carbon::now()->subDays(7))
                    ->orderBy('expense_date', 'desc')
                    ->limit(5)
                    ->get()
            ]);
        }
        
        // For cashiers, add their personal sales data
        if (!$user->can('view-reports') && $user->can('process-sales')) {
            $dashboardData['mySalesToday'] = Sale::withoutTrashed() // Exclude soft-deleted sales
                ->where('tenant_id', $tenantId)
                ->where('user_id', $user->id) // Assuming user_id is the correct column
                ->whereDate('created_at', $today) // Assuming created_at is the correct column
                ->sum('total_amount');
            
            $dashboardData['mySalesCountToday'] = Sale::withoutTrashed() // Exclude soft-deleted sales
                ->where('tenant_id', $tenantId)
                ->where('user_id', $user->id) // Assuming user_id is the correct column
                ->whereDate('created_at', $today) // Assuming created_at is the correct column
                ->count();
        }
        
        return Inertia::render('Dashboard', [
            'dashboardData' => $dashboardData
        ]);
    }
}
