<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Client;
use App\Models\Product;
use App\Models\Sale;
use App\Models\SaleItem;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;

class SaleFinalizationController extends Controller
{
    /**
     * Finalize a sale
     * 
     * @param Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function __invoke(Request $request)
    {
        // Validate the request
        $validated = $request->validate([
            'client_id' => 'required|exists:clients,id',
            'items' => 'required|array|min:1',
            'items.*.product_id' => 'required|exists:products,id',
            'items.*.quantity' => 'required|numeric|min:0.01',
            'items.*.selling_price' => 'required|numeric|min:0',
            'notes' => 'nullable|string',
            'discount_amount' => 'nullable|numeric|min:0',
        ]);

        // Start a database transaction
        return DB::transaction(function () use ($validated, $request) {
            $client = Client::findOrFail($validated['client_id']);
            
            // Check if client belongs to the same tenant
            if ($client->tenant_id !== Auth::user()->tenant_id) {
                throw ValidationException::withMessages([
                    'client_id' => ['Client not found.']
                ]);
            }
            
            // Create the sale
            $sale = new Sale([
                'tenant_id' => Auth::user()->tenant_id,
                'user_id' => Auth::id(),
                'client_id' => $client->id,
                'total_amount' => 0, // Will be calculated below
                'discount_amount' => 0, // Will be set below
                'final_amount' => 0, // Will be calculated below
                'notes' => $validated['notes'] ?? null,
            ]);
            
            $sale->save();
            
            $totalAmount = 0;
            
            // Process each item
            foreach ($validated['items'] as $item) {
                $product = Product::where('id', $item['product_id'])
                    ->where('tenant_id', Auth::user()->tenant_id)
                    ->lockForUpdate()
                    ->firstOrFail();
                
                // Check if there's enough stock
                if ($product->quantity < $item['quantity']) {
                    throw ValidationException::withMessages([
                        'items' => ["Not enough stock for product: {$product->name}"]
                    ]);
                }
                
                // Create sale item
                $saleItem = new SaleItem([
                    'tenant_id' => Auth::user()->tenant_id,
                    'sale_id' => $sale->id,
                    'product_id' => $product->id,
                    'quantity' => $item['quantity'],
                    'price_at_sale' => $item['selling_price'], // Use the selling price from the request
                ]);
                
                $saleItem->save();
                
                // Update product stock
                $product->quantity -= $item['quantity'];
                $product->save();
                
                // Calculate item total and add to running total
                $itemTotal = (float)$item['selling_price'] * (float)$item['quantity'];
                $totalAmount += $itemTotal;
            }
            
            // Get the discount amount from the request (default to 0)
            $discountAmount = isset($validated['discount_amount']) ? (float)$validated['discount_amount'] : 0;
            
            // Ensure discount doesn't exceed the total amount
            $discountAmount = min($discountAmount, $totalAmount);
            
            // Calculate the final amount after discount
            $finalAmount = $totalAmount - $discountAmount;
            
            // Update the sale with totals and discount
            $sale->total_amount = $totalAmount;
            $sale->discount_amount = $discountAmount;
            $sale->final_amount = $finalAmount;
            $sale->save();
            
            // Update client balance with the FINAL amount (after discount)
            // CRUCIAL REQUIREMENT: Payments are tracked against Client balances
            $client->balance += $finalAmount;
            $client->save();
            
            return response()->json([
                'message' => 'Sale completed successfully',
                'sale' => $sale,
            ]);
        });
    }
}
