<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;

class UpdateSaleTotals extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:update-sale-totals {--tenant=all : The tenant ID to update sales for (default: all tenants)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Update all sales with zero totals to have the correct calculated amount based on their sale items';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $tenantOption = $this->option('tenant');
        
        $this->info('Starting to update sale totals...');
        
        // Get the query builder for sales with zero totals
        $query = \App\Models\Sale::where('total_amount', 0)
            ->orWhere('total_amount', null);
        
        // Filter by tenant if specified
        if ($tenantOption !== 'all') {
            $query->where('tenant_id', $tenantOption);
            $this->info("Filtering for tenant ID: {$tenantOption}");
        }
        
        // Get the count of sales to update
        $count = $query->count();
        $this->info("Found {$count} sales with zero totals to update.");
        
        if ($count === 0) {
            $this->info('No sales need updating. Exiting.');
            return 0;
        }
        
        // Create a progress bar
        $bar = $this->output->createProgressBar($count);
        $bar->start();
        
        // Process in chunks to avoid memory issues with large datasets
        $query->with('saleItems')->chunk(100, function ($sales) use ($bar) {
            foreach ($sales as $sale) {
                // Calculate the total from sale items
                $total = 0;
                foreach ($sale->saleItems as $item) {
                    $total += (float)$item->price_at_sale * (int)$item->quantity;
                }
                
                // Update the sale with the calculated total
                if ($total > 0) {
                    $sale->total_amount = round($total, 2);
                    $sale->save();
                }
                
                $bar->advance();
            }
        });
        
        $bar->finish();
        $this->newLine(2);
        $this->info('Sale totals have been updated successfully!');
        
        return 0;
    }
}
