<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Supplier;
use App\Models\Purchase;
use App\Models\PurchaseReturn;
use App\Models\Payment;
use Illuminate\Support\Facades\DB;

class RecalculateSupplierBalances extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:recalculate-supplier-balances {--dry-run : Show what would be changed without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Recalculate supplier balances based on actual transactions';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        
        if ($dryRun) {
            $this->info('Running in dry-run mode. No changes will be made.');
        }
        
        $suppliers = Supplier::all();
        $this->info("Processing {$suppliers->count()} suppliers...");
        
        $correctedCount = 0;
        
        foreach ($suppliers as $supplier) {
            // Calculate correct balance
             $purchases = Purchase::where('supplier_id', $supplier->id)->sum('total_amount');
             $returns = PurchaseReturn::where('supplier_id', $supplier->id)->sum('returned_amount');
            
            // Only count valid payments (exclude unpaid checks)
            $payments = Payment::where('paymentable_type', Supplier::class)
                ->where('paymentable_id', $supplier->id)
                ->get()
                ->filter(function ($payment) {
                    return $payment->shouldCountInTotals();
                })
                ->sum('amount');
            
            // Balance = Purchases - Returns - Valid Payments
            $correctBalance = $purchases - $returns - $payments;
            
            if (abs($supplier->balance - $correctBalance) > 0.01) { // Allow for small floating point differences
                $this->line(sprintf(
                    'Supplier %s (ID: %d): Current balance: %s, Correct balance: %s',
                    $supplier->name,
                    $supplier->id,
                    number_format($supplier->balance, 2),
                    number_format($correctBalance, 2)
                ));
                
                if (!$dryRun) {
                    $supplier->balance = $correctBalance;
                    $supplier->save();
                    $this->info('  ✓ Updated');
                } else {
                    $this->info('  → Would be updated');
                }
                
                $correctedCount++;
            }
        }
        
        if ($correctedCount === 0) {
            $this->info('All supplier balances are correct!');
        } else {
            if ($dryRun) {
                $this->info("Found {$correctedCount} suppliers with incorrect balances.");
                $this->info('Run without --dry-run to apply the corrections.');
            } else {
                $this->info("Corrected {$correctedCount} supplier balances.");
            }
        }
        
        return 0;
    }
}
