<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\Client;
use App\Models\Sale;
use App\Models\SaleReturn;
use App\Models\Payment;
use Illuminate\Support\Facades\DB;

class RecalculateClientBalances extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:recalculate-client-balances {--dry-run : Show what would be changed without making changes}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Recalculate client balances based on actual transactions';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $dryRun = $this->option('dry-run');
        
        if ($dryRun) {
            $this->info('Running in dry-run mode. No changes will be made.');
        }
        
        $clients = Client::all();
        $this->info("Processing {$clients->count()} clients...");
        
        $correctedCount = 0;
        
        foreach ($clients as $client) {
            // Calculate correct balance
            $sales = Sale::where('client_id', $client->id)->sum('final_amount');
            $returns = SaleReturn::where('client_id', $client->id)->sum('returned_amount');
            
            // Only count valid payments (exclude unpaid checks)
            $payments = Payment::where('paymentable_type', Client::class)
                ->where('paymentable_id', $client->id)
                ->get()
                ->filter(function ($payment) {
                    return $payment->shouldCountInTotals();
                })
                ->sum('amount');
            
            // Balance = Sales - Returns - Valid Payments
            $correctBalance = $sales - $returns - $payments;
            
            if (abs($client->balance - $correctBalance) > 0.01) { // Allow for small floating point differences
                $this->line(sprintf(
                    'Client %s (ID: %d): Current balance: %s, Correct balance: %s',
                    $client->name,
                    $client->id,
                    number_format($client->balance, 2),
                    number_format($correctBalance, 2)
                ));
                
                if (!$dryRun) {
                    $client->balance = $correctBalance;
                    $client->save();
                    $this->info('  ✓ Updated');
                } else {
                    $this->info('  → Would be updated');
                }
                
                $correctedCount++;
            }
        }
        
        if ($correctedCount === 0) {
            $this->info('All client balances are correct!');
        } else {
            if ($dryRun) {
                $this->info("Found {$correctedCount} clients with incorrect balances.");
                $this->info('Run without --dry-run to apply the corrections.');
            } else {
                $this->info("Corrected {$correctedCount} client balances.");
            }
        }
        
        return 0;
    }
}
