<?php

namespace App\Console\Commands;

use App\Models\InventoryLayer;
use App\Models\Product;
use App\Models\Purchase;
use App\Models\PurchaseItem;
use App\Models\Sale;
use App\Models\SaleItem;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;

class MigrateToFifoInventory extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'inventory:migrate-to-fifo';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Migrate existing inventory data to FIFO inventory valuation system';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Starting migration to FIFO inventory system...');

        // Step 1: Create inventory layers from existing purchases
        $this->createInventoryLayers();

        // Step 2: Update sale items with cost information
        $this->updateSaleItemCosts();

        // Step 3: Update sales with profitability metrics
        $this->updateSaleProfitability();

        $this->info('Migration to FIFO inventory system completed successfully!');
        return 0;
    }

    /**
     * Create inventory layers from existing purchases.
     */
    private function createInventoryLayers()
    {
        $this->info('Creating inventory layers from existing purchases...');
        $bar = $this->output->createProgressBar(PurchaseItem::count());
        
        // Process each purchase item
        PurchaseItem::with(['purchase', 'product'])->chunk(100, function ($purchaseItems) use ($bar) {
            foreach ($purchaseItems as $purchaseItem) {
                // Skip if product or purchase is missing
                if (!$purchaseItem->product || !$purchaseItem->purchase) {
                    $bar->advance();
                    continue;
                }
                
                // Calculate how much of this purchase item might still be in inventory
                $product = $purchaseItem->product;
                
                // Create inventory layer
                InventoryLayer::create([
                    'tenant_id' => $purchaseItem->tenant_id,
                    'product_id' => $purchaseItem->product_id,
                    'purchase_id' => $purchaseItem->purchase_id,
                    'purchase_item_id' => $purchaseItem->id,
                    'quantity_initial' => $purchaseItem->quantity,
                    'quantity_remaining' => min($purchaseItem->quantity, $product->quantity), // Estimate remaining
                    'unit_cost' => $purchaseItem->cost_price,
                    'created_at' => $purchaseItem->purchase->created_at ?? now(),
                ]);
                
                $bar->advance();
            }
        });
        
        $bar->finish();
        $this->newLine();
    }

    /**
     * Update sale items with cost information.
     */
    private function updateSaleItemCosts()
    {
        $this->info('Updating sale items with cost information...');
        $bar = $this->output->createProgressBar(SaleItem::count());
        
        // Process each sale item
        SaleItem::with(['product', 'sale'])->chunk(100, function ($saleItems) use ($bar) {
            foreach ($saleItems as $saleItem) {
                // Skip if product is missing
                if (!$saleItem->product) {
                    $bar->advance();
                    continue;
                }
                
                // Use current purchase price as an estimate for historical cost
                $costAtSale = $saleItem->product->purchase_price;
                $unitProfit = $saleItem->price_at_sale - $costAtSale;
                $totalCost = $costAtSale * $saleItem->quantity;
                $totalProfit = $unitProfit * $saleItem->quantity;
                
                // Update the sale item
                $saleItem->update([
                    'cost_at_sale' => $costAtSale,
                    'unit_profit' => $unitProfit,
                    'total_cost' => $totalCost,
                    'total_profit' => $totalProfit,
                ]);
                
                $bar->advance();
            }
        });
        
        $bar->finish();
        $this->newLine();
    }

    /**
     * Update sales with profitability metrics.
     */
    private function updateSaleProfitability()
    {
        $this->info('Updating sales with profitability metrics...');
        $bar = $this->output->createProgressBar(Sale::count());
        
        // Process each sale
        Sale::with('saleItems')->chunk(100, function ($sales) use ($bar) {
            foreach ($sales as $sale) {
                // Calculate total cost from sale items
                $totalCost = $sale->saleItems->sum('total_cost');
                
                // Calculate effective amount (final_amount if available, otherwise total_amount)
                $effectiveAmount = $sale->final_amount ?? $sale->total_amount;
                
                // Calculate gross profit
                $grossProfit = $effectiveAmount - $totalCost;
                
                // Calculate profit margin percentage
                $profitMarginPercentage = ($effectiveAmount > 0) ? (($grossProfit / $effectiveAmount) * 100) : 0;
                
                // Update the sale
                $sale->update([
                    'total_cost' => $totalCost,
                    'gross_profit' => $grossProfit,
                    'profit_margin_percentage' => $profitMarginPercentage,
                ]);
                
                $bar->advance();
            }
        });
        
        $bar->finish();
        $this->newLine();
    }
}
